<?php
// Configurações de exibição de erro
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

// Caminho para os assets e inclusão de arquivos de sessão e layout
$static_url = dirname($_SERVER['SCRIPT_NAME']) . '/assets/';
include 'session_check.php';

$page_title = "MAC Users";

$page_content = '
<div class="card">
    <div class="card-header border-0 pt-6">
        <div class="card-title">
            <div class="d-flex align-items-center position-relative my-1">
                <i class="ki-outline ki-magnifier fs-3 position-absolute ms-5"></i>
                <input type="text" id="search_playlist" class="form-control form-control-solid w-250px ps-12" placeholder="Search MACs..." />
            </div>
        </div>
        <div class="card-toolbar">
            <div class="d-flex justify-content-end">
                <button type="button" class="btn btn-primary" id="add-mac-btn">Add New MAC</button>
            </div>
        </div>
    </div>
    <div class="card-body pt-0">
        <table class="table align-middle table-row-dashed fs-6 gy-5" id="mac_table">
            <thead>
                <tr class="text-start text-gray-500 fw-bold fs-7 text-uppercase gs-0">
                    <th class="min-w-125px">ID</th>
                    <th class="min-w-125px">DNS</th>
                    <th class="min-w-125px">MAC Address</th>
                    <th class="min-w-125px">Username</th>
                    <th class="min-w-125px">Password</th>
                    <th class="min-w-125px">PIN</th>
                    <th class="min-w-125px text-end">Actions</th>
                </tr>
            </thead>
            <tbody class="fw-semibold text-gray-600"></tbody>
        </table>
    </div>
</div>

<div class="modal fade" id="mac_modal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-dialog-centered mw-650px">
        <div class="modal-content">
            <form id="mac_form" class="form">
                <div class="modal-header">
                    <h2 id="modal_title" class="fw-bold">Add MAC User</h2>
                    <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
                </div>
                <div class="modal-body py-10 px-lg-17">
                    <input type="hidden" name="mac_id" id="mac_id">
                    <div class="mb-7">
                        <label for="dns_id" class="fs-6 fw-semibold mb-2">DNS</label>
                        <select class="form-select fw-bold" name="dns_id" id="dns_id" data-control="select2" data-placeholder="Select a DNS..." required>
                            <option></option>
                        </select>
                    </div>
                    <div class="mb-7">
                        <label for="mac_address" class="fs-6 fw-semibold mb-2">MAC Address</label>
                        <input type="text" class="form-control" name="mac_address" id="mac_address" placeholder="Enter MAC Address" required>
                    </div>
                    <div class="mb-7">
                        <label for="username" class="fs-6 fw-semibold mb-2">Username</label>
                        <input type="text" class="form-control" name="username" id="username" placeholder="Enter Username" required>
                    </div>
                    <div class="mb-7">
                        <label for="password" class="fs-6 fw-semibold mb-2">Password</label>
                        <input type="password" class="form-control" name="password" id="password" placeholder="Enter Password" required>
                    </div>
                    <div class="mb-7">
                        <label for="pin" class="fs-6 fw-semibold mb-2">PIN</label>
                        <input type="text" class="form-control" name="pin" id="pin" placeholder="Enter PIN" required>
                    </div>
                </div>
                <div class="modal-footer">
                    <button type="button" class="btn btn-light" data-bs-dismiss="modal">Cancel</button>
                    <button type="submit" class="btn btn-primary" id="save-mac-btn">Save</button>
                </div>
            </form>
        </div>
    </div>
</div>
';

// Inclui o layout e o script JS
include 'includes/layout.php'; ?>

<script>
    let macTable;
    let macModal;
    let dnsOptionsCache = {};

    const macForm = document.getElementById('mac_form');
    const modalTitle = document.getElementById('modal_title');
    const saveBtn = document.getElementById('save-mac-btn');
    const addBtn = document.getElementById('add-mac-btn');

    document.addEventListener("DOMContentLoaded", async () => {
        macModal = new bootstrap.Modal(document.getElementById("mac_modal"));

        // Carrega opções de DNS em cache antes de inicializar a tabela
        await loadDnsOptions();

        macTable = $('#mac_table').DataTable({
            processing: true,
            serverSide: false,
            ajax: {
                url: "actions/mac_actions.php",
                type: "POST",
                data: { action: "view" },
                dataSrc: function(json) {
                    if (json.success && Array.isArray(json.data)) {
                        return json.data;
                    }
                    console.error("Erro no retorno da API:", json.message);
                    return [];
                }
            },
            columns: [
                { data: 'id' },
                { data: 'dns_id', render: function(data, type, row) {
                    return dnsOptionsCache[data] || data;
                }},
                { data: 'mac_address' },
                { data: 'username' },
                { data: null, render: function() {
                    return '********';
                }},
                { data: 'pin' },
                { data: null, orderable: false, render: function(data, type, row) {
                    return `
                        <div class="text-end">
                           
                            <button class="btn btn-sm btn-danger" onclick="deleteMac(${row.id})">Delete</button>
                        </div>
                    `;
                }}
            ],
            initComplete: function() {
                const searchInput = document.getElementById('search_playlist');
                if (searchInput) {
                    $(searchInput).on('keyup', function() {
                        macTable.search(this.value).draw();
                    });
                }
            }
        });

        addBtn.addEventListener('click', openAddModal);
        macForm.addEventListener('submit', async (e) => {
            e.preventDefault();
            saveMac();
        });
    });

    async function loadDnsOptions() {
        if (Object.keys(dnsOptionsCache).length > 0) {
            return;
        }
        try {
            const response = await fetch("actions/mac_actions.php", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ action: "get_dns_options" }),
            });
            const data = await response.json();
            if (data.success) {
                data.data.forEach(dns => {
                    dnsOptionsCache[dns.id] = dns.title;
                });
            }
        } catch (error) {
            console.error("Error loading DNS options:", error);
        }
    }

    function populateDnsDropdown(selectedValue = "") {
        const dnsDropdown = document.getElementById("dns_id");
        $(dnsDropdown).empty().append('<option value="">Select DNS</option>');
        for (const id in dnsOptionsCache) {
            const newOption = new Option(dnsOptionsCache[id], id, false, id == selectedValue);
            $(dnsDropdown).append(newOption).trigger('change');
        }
        $(dnsDropdown).select2({ dropdownParent: $('#mac_modal') });
    }

    function openAddModal() {
        modalTitle.textContent = "Add MAC User";
        macForm.reset();
        document.getElementById("mac_id").value = "";
        populateDnsDropdown();
        macModal.show();
    }

    async function openEditModal(id) {
        try {
            const response = await fetch("actions/mac_actions.php", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify({ action: "get_single_mac", id: id }),
            });
            const data = await response.json();

            if (data.success) {
                const record = data.data;
                modalTitle.textContent = "Edit MAC User";
                document.getElementById("mac_id").value = record.id;
                document.getElementById("mac_address").value = record.mac_address;
                document.getElementById("username").value = record.username;
                document.getElementById("password").value = record.password;
                document.getElementById("pin").value = record.pin;
                populateDnsDropdown(record.dns_id);
                macModal.show();
            } else {
                Swal.fire("Error", data.message, "error");
            }
        } catch (error) {
            console.error("Error fetching record for editing:", error);
            Swal.fire("Error", "Failed to fetch record data.", "error");
        }
    }

    async function saveMac() {
        saveBtn.disabled = true;

        const id = document.getElementById("mac_id").value;
        const dns_id = document.getElementById("dns_id").value.trim();
        const mac_address = document.getElementById("mac_address").value.trim();
        const username = document.getElementById("username").value.trim();
        const password = document.getElementById("password").value.trim();
        const pin = document.getElementById("pin").value.trim();

        if (!dns_id || !mac_address || !username || !password || !pin) {
            Swal.fire("Error", "All fields are required!", "error");
            saveBtn.disabled = false;
            return;
        }

        if (!validateMACAddress(mac_address)) {
            Swal.fire("Error", "Invalid MAC Address!", "error");
            saveBtn.disabled = false;
            return;
        }

        const action = id ? "edit" : "add";
        const payload = { action, id, dns_id, mac_address, username, password, pin };

        try {
            const response = await fetch("actions/mac_actions.php", {
                method: "POST",
                headers: { "Content-Type": "application/json" },
                body: JSON.stringify(payload),
            });
            const data = await response.json();

            Swal.fire(data.success ? "Success" : "Error", data.message, data.success ? "success" : "error");
            if (data.success) {
                macModal.hide();
                macTable.ajax.reload();
            }
        } catch (error) {
            console.error("Error saving MAC:", error);
            Swal.fire("Error", "Failed to save data.", "error");
        } finally {
            saveBtn.disabled = false;
        }
    }

    function deleteMac(id) {
        Swal.fire({
            title: "Are you sure?",
            text: "You won't be able to revert this!",
            icon: "warning",
            showCancelButton: true,
            confirmButtonText: "Yes, delete it!",
            cancelButtonText: "Cancel"
        }).then(async (result) => {
            if (result.isConfirmed) {
                try {
                    const response = await fetch("actions/mac_actions.php", {
                        method: "POST",
                        headers: { "Content-Type": "application/json" },
                        body: JSON.stringify({ action: "delete", id }),
                    });
                    const data = await response.json();
                    Swal.fire(data.success ? "Deleted!" : "Error", data.message, data.success ? "success" : "error");
                    if (data.success) {
                        macTable.ajax.reload();
                    }
                } catch (error) {
                    console.error("Error deleting MAC:", error);
                    Swal.fire("Error", "Failed to delete record.", "error");
                }
            }
        });
    }

    function validateMACAddress(mac) {
        const regex = /^([0-9A-Fa-f]{2}[:-]){5}([0-9A-Fa-f]{2})$/;
        return regex.test(mac);
    }
</script>