<?php
error_reporting(E_ALL);
ini_set('display_errors', 1);
ini_set('display_startup_errors', 1);

include 'session_check.php';

$static_url = dirname($_SERVER['SCRIPT_NAME']) . '/rtx/';
$adTypeFile = __DIR__ . '/api/ad_type.json';
$currentAdType = json_decode(@file_get_contents($adTypeFile), true)['adType'] ?? 'manual';

// Atualização via POST
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json');

    $newAdType = $_POST['ad_type'] ?? '';
    $allowedTypes = ['manual', 'tmdb', 'sport'];

    if (in_array($newAdType, $allowedTypes)) {
        file_put_contents($adTypeFile, json_encode(['adType' => $newAdType], JSON_PRETTY_PRINT));
        echo json_encode(['status' => 'success', 'message' => 'Tipo de anúncio atualizado com sucesso!']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Tipo de anúncio inválido.']);
    }
    exit;
}

$page_title = "Seleção de Tipo de Anúncio";

$adTypes = [
    'manual' => [
        'title' => 'Banner Manual',
        'desc' => 'Anúncios personalizados criados manualmente',
        'img' => 'manual.png'
    ],
    'tmdb' => [
        'title' => 'Filmes Automáticos',
        'desc' => 'Anúncios baseados no banco de dados do TMDB',
        'img' => 'tmdb.png'
    ],
    'sport' => [
        'title' => 'Guia Esportivo',
        'desc' => 'Anúncios relacionados a eventos esportivos',
        'img' => 'sport.png'
    ]
];

ob_start();
?>

<!-- Dependências -->
<script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/sweetalert2@11"></script>

<div class="card">
    <div class="card-header">
        <h3 class="card-title">Selecione o Tipo de Anúncio</h3>
    </div>
    <div class="card-body">
        <div class="row">
            <?php foreach ($adTypes as $type => $data): ?>
                <div class="col-md-4 mb-5 mb-md-0">
                    <div class="card card-custom ad-type-card <?= $currentAdType === $type ? 'card-active' : '' ?>" data-type="<?= htmlspecialchars($type) ?>">
                        <div class="card-body text-center">
                            <div class="symbol symbol-100px mb-7">
                                <img src="<?= $static_url . 'img/' . htmlspecialchars($data['img']) ?>" class="img-fluid" alt="<?= htmlspecialchars($data['title']) ?>">
                            </div>
                            <h4 class="text-dark mb-4"><?= htmlspecialchars($data['title']) ?></h4>
                            <p class="text-muted"><?= htmlspecialchars($data['desc']) ?></p>
                            <button class="btn btn-primary select-ad-type" data-type="<?= htmlspecialchars($type) ?>">
                                <?= $currentAdType === $type ? 'Ativo' : 'Selecionar' ?>
                            </button>
                        </div>
                    </div>
                </div>
            <?php endforeach; ?>
        </div>
    </div>
</div>

<!-- Estilo -->
<style>
    .ad-type-card {
        transition: all 0.3s ease;
        cursor: pointer;
        height: 100%;
    }
    .ad-type-card:hover {
        transform: translateY(-5px);
        box-shadow: 0 5px 15px rgba(0,0,0,0.1);
    }
    .card-active {
        border: 2px solid #00bfff;
        background-color: #0d1117;
        color: #e0f7ff;
    }
    .select-ad-type {
        width: 120px;
    }
</style>

<!-- Script -->
<script>
$(document).ready(function() {
    $(".ad-type-card").click(function() {
        const adType = $(this).data("type");
        selectAdType(adType);
    });

    $(".select-ad-type").click(function(e) {
        e.stopPropagation();
        const adType = $(this).data("type");
        selectAdType(adType);
    });

    function selectAdType(adType) {
        Swal.fire({
            title: "Confirmar alteração",
            text: "Deseja realmente alterar o tipo de anúncio para " + getAdTypeName(adType) + "?",
            icon: "question",
            showCancelButton: true,
            confirmButtonText: "Sim, alterar",
            cancelButtonText: "Cancelar"
        }).then((result) => {
            if (result.isConfirmed) {
                $.ajax({
                    url: window.location.href,
                    method: "POST",
                    data: { ad_type: adType },
                    dataType: "json",
                    success: function(response) {
                        if (response.status === "success") {
                            Swal.fire("Sucesso!", response.message, "success").then(() => {
                                location.reload();
                            });
                        } else {
                            Swal.fire("Erro", response.message, "error");
                        }
                    },
                    error: function(xhr, status, error) {
                        Swal.fire("Erro", "Erro ao tentar atualizar: " + error, "error");
                    }
                });
            }
        });
    }

    function getAdTypeName(type) {
        const types = {
            'manual': 'Banner Manual',
            'tmdb': 'Filmes Automáticos',
            'sport': 'Guia Esportivo'
        };
        return types[type] || type;
    }
});
</script>

<?php
$page_content = ob_get_clean();
include 'includes/layout.php';
